%%%-------------------------------------------------------------------
%%% @author Piotr Duleba
%%% @copyright (C) 2020 ACK CYFRONET AGH
%%% This software is released under the MIT license
%%% cited in 'LICENSE.txt'.
%%% @end
%%%-------------------------------------------------------------------
%%% @doc
%%% This file provides functions for making rpc calls to OzWorker Nodes for use during tests.
%%% @end
%%%-------------------------------------------------------------------
-module(ozw_test_rpc).
-author("Piotr Duleba").

-include_lib("ctool/include/test/test_utils.hrl").

%% API
-export([
    call/1, call/2,
    insecure_call/1, insecure_call/2,

    call/3, call/4,
    insecure_call/3, insecure_call/4,

    simulate_downtime/1,
    timestamp_seconds/0,
    timestamp_seconds/1,
    get_env/1,
    set_env/2,
    get_domain/0,

    add_user_to_cluster/3,

    list_users/0,
    list_users/1,
    create_user/0,
    create_user/1,
    get_user_protected_data/1,
    get_user_protected_data/2,
    create_user_temporary_token/3,
    create_user_temporary_token/4,
    create_user_temporary_access_token/2,
    are_basic_credentials_valid/2,

    list_groups/1,
    create_group/1,
    get_group_protected_data/3,
    get_group_users/1,

    list_spaces/0,
    list_spaces/1,
    create_space/2,
    add_user_to_space/2,
    create_space_support_token/2,
    get_space_protected_data/2,
    get_space_protected_data/3,
    space_set_user_privileges/3,
    update_space_name/2,
    delete_space/1,

    create_inventory_for_user/3,
    add_user_to_inventory/4,
    create_lambda/2,
    create_workflow_schema/2,
    update_workflow_schema/3,
    get_workflow_schema/2,

    list_handle_services/0,
    add_user_to_handle_service/3,
    remove_user_from_handle_service/2
]).

-define(CALL(NodeSelector, Args), test_rpc:call(oz_worker, NodeSelector, test_rpc_api, ?FUNCTION_NAME, Args)).

-define(MULTICALL(Args), lists:foreach(fun(Node) -> ?CALL(Node, Args) end, oct_background:get_zone_nodes())).

%%%===================================================================
%%% API functions
%%%===================================================================


-spec call(test_rpc:routine()) ->
    term() | no_return().
call(Routine) ->
    call(Routine, infinity).


-spec call(test_rpc:routine(), timeout()) ->
    term() | no_return().
call(Routine, Timeout) ->
    test_rpc:call(oz_worker, zone, Routine, Timeout).


-spec insecure_call(test_rpc:routine()) ->
    term() | no_return().
insecure_call(Routine) ->
    insecure_call(Routine, infinity).


-spec insecure_call(test_rpc:routine(), timeout()) ->
    term() | no_return().
insecure_call(Routine, Timeout) ->
    test_rpc:insecure_call(oz_worker, zone, Routine, Timeout).


-spec call(module(), atom(), [term()]) ->
    term() | no_return().
call(Module, FunctionName, Args) ->
    call(Module, FunctionName, Args, infinity).


-spec call(module(), atom(), [term()], timeout()) ->
    term() | no_return().
call(Module, FunctionName, Args, Timeout) ->
    test_rpc:call(oz_worker, zone, Module, FunctionName, Args, Timeout).


-spec insecure_call(module(), atom(), [term()]) ->
    term() | no_return().
insecure_call(Module, FunctionName, Args) ->
    insecure_call(Module, FunctionName, Args, infinity).


-spec insecure_call(module(), atom(), [term()], timeout()) ->
    term() | no_return().
insecure_call(Module, FunctionName, Args, Timeout) ->
    test_rpc:insecure_call(oz_worker, zone, Module, FunctionName, Args, Timeout).


-spec simulate_downtime(time:seconds()) -> ok.
simulate_downtime(Seconds) ->
    ?assertMatch(ok, ?CALL(zone, [Seconds])).


-spec timestamp_seconds() -> time:seconds().
timestamp_seconds() ->
    timestamp_seconds(zone).


-spec timestamp_seconds(oct_background:node_selector()) -> time:seconds().
timestamp_seconds(NodeSelector) ->
    ?CALL(NodeSelector, []).


-spec get_env(Key :: atom()) -> term().
get_env(Env) ->
    ?CALL(zone, [Env]).


-spec set_env(Key :: atom(), term()) -> ok.
set_env(Env, Value) ->
    ?MULTICALL([Env, Value]).


-spec get_domain() -> binary().
get_domain() ->
    ?CALL(zone, []).


-spec add_user_to_cluster(binary(), binary(), [term()]) -> ok.
add_user_to_cluster(ClusterId, UserId, Privileges) ->
    ?assertMatch({ok, _}, ?CALL(zone, [aai:root_auth(), ClusterId, UserId, Privileges])),
    ok.


-spec list_users() -> [binary()].
list_users() ->
    list_users(zone).


-spec list_users(oct_background:node_selector()) -> [binary()].
list_users(NodeSelector) ->
    {ok, Users} = ?assertMatch({ok, _}, ?CALL(NodeSelector, [aai:root_auth()])),
    Users.


-spec create_user() -> binary().
create_user() ->
    {ok, UserId} = ?assertMatch({ok, _}, ?CALL(zone, [aai:root_auth()])),
    UserId.


create_user(Data) ->
    {ok, UserId} = ?assertMatch({ok, _}, ?CALL(zone, [aai:root_auth(), Data])),
    UserId.


-spec get_user_protected_data(binary()) -> map().
get_user_protected_data(UserId) ->
    get_user_protected_data(zone, UserId).


-spec get_user_protected_data(oct_background:node_selector(), binary()) -> map().
get_user_protected_data(NodeSelector, UserId) ->
    {ok, UserDetails} = ?assertMatch({ok, _}, ?CALL(NodeSelector, [aai:root_auth(), UserId])),
    UserDetails.


-spec create_user_temporary_token(aai:auth(), binary(), map()) -> tokens:token().
create_user_temporary_token(Auth, UserId, Data) ->
    create_user_temporary_token(zone, Auth, UserId, Data).


-spec create_user_temporary_token(oct_background:node_selector(), aai:auth(), binary(), map()) -> tokens:token().
create_user_temporary_token(NodeSelector, Auth, UserId, Data) ->
    {ok, Token} = ?assertMatch({ok, _}, ?CALL(NodeSelector, [Auth, UserId, Data])),
    Token.


-spec create_user_temporary_access_token(oct_background:node_selector(), binary()) -> tokens:token().
create_user_temporary_access_token(NodeSelector, UserId) ->
    {ok, Token} = ?assertMatch({ok, _}, ?CALL(NodeSelector, [UserId])),
    Token.


-spec are_basic_credentials_valid(binary(), binary()) -> boolean().
are_basic_credentials_valid(UserName, Password) ->
    ?CALL(zone, [UserName, Password]).


-spec list_groups(oct_background:node_selector()) -> [binary()].
list_groups(NodeSelector) ->
    {ok, GroupIds} = ?assertMatch({ok, _}, ?CALL(NodeSelector, [aai:root_auth()])),
    GroupIds.


-spec create_group(binary()) -> binary().
create_group(GroupName) ->
    {ok, GroupId} = ?assertMatch({ok, _}, ?CALL(zone, [aai:root_auth(), GroupName])),
    GroupId.


-spec get_group_protected_data(oct_background:node_selector(), aai:auth(), binary()) -> map().
get_group_protected_data(NodeSelector, Auth, GroupId) ->
    {ok, Data} = ?assertMatch({ok, _}, ?CALL(NodeSelector, [Auth, GroupId])),
    Data.


-spec get_group_users(binary()) -> [binary()].
get_group_users(GroupId) ->
    {ok, Users} = ?assertMatch({ok, _}, ?CALL(zone, [aai:root_auth(), GroupId])),
    Users.


-spec create_space(binary(), binary()) -> binary().
create_space(UserId, SpaceName) ->
    {ok, SpaceId} = ?assertMatch({ok, _}, ?CALL(zone, [aai:user_auth(UserId), UserId, SpaceName])),
    SpaceId.


-spec add_user_to_space(binary(), binary()) -> ok.
add_user_to_space(SpaceId, UserId) ->
    ?assertMatch({ok, _}, ?CALL(zone, [aai:root_auth(), SpaceId, UserId])),
    ok.


-spec create_space_support_token(binary(), binary()) -> binary().
create_space_support_token(UserId, SpaceId) ->
    {ok, Token} = ?assertMatch({ok, _}, ?CALL(zone, [aai:user_auth(UserId), SpaceId])),
    {ok, SerializedToken} = tokens:serialize(Token),
    SerializedToken.


-spec list_spaces() -> [binary()].
list_spaces() ->
    list_spaces(zone).


-spec list_spaces(oct_background:node_selector()) -> [binary()].
list_spaces(NodeSelector) ->
    {ok, Spaces} = ?assertMatch({ok, _}, ?CALL(NodeSelector, [aai:root_auth()])),
    Spaces.


-spec get_space_protected_data(aai:auth(), binary()) -> map().
get_space_protected_data(Auth, SpaceId) ->
    get_space_protected_data(zone, Auth, SpaceId).


-spec get_space_protected_data(oct_background:node_selector(), aai:auth(), binary()) -> map().
get_space_protected_data(NodeSelector, Auth, SpaceId) ->
    {ok, Data} = ?assertMatch({ok, _}, ?CALL(NodeSelector, [Auth, SpaceId])),
    Data.


-spec space_set_user_privileges(binary(), binary(), [privileges:space_privilege()]) ->
    ok.
space_set_user_privileges(SpaceId, UserId, Privileges) ->
    ?assertMatch(ok, ?CALL(zone, [aai:root_auth(), SpaceId, UserId, Privileges])).



-spec update_space_name(binary(), binary()) -> ok.
update_space_name(SpaceId, NewName) ->
    ?assertMatch(ok, ?CALL(zone, [aai:root_auth(), SpaceId, NewName])).


-spec delete_space(binary()) -> ok.
delete_space(SpaceId) ->
    ?assertMatch(ok, ?CALL(zone, [aai:root_auth(), SpaceId])).


-spec create_inventory_for_user(aai:auth(), binary(), term()) -> binary().
create_inventory_for_user(Auth, UserId, NameOrData) ->
    {ok, Id} = ?assertMatch({ok, _}, ?CALL(zone, [Auth, UserId, NameOrData])),
    Id.


-spec add_user_to_inventory(aai:auth(), binary(), binary(), [privileges:atm_inventory_privilege()]) ->
    binary().
add_user_to_inventory(Auth, AtmInventoryId, UserId, Privileges) ->
    {ok, Id} = ?assertMatch({ok, _}, ?CALL(zone, [Auth, AtmInventoryId, UserId, Privileges])),
    Id.


-spec create_lambda(aai:auth(), term()) -> binary().
create_lambda(Auth, Data) ->
    {ok, Id} = ?assertMatch({ok, _}, ?CALL(zone, [Auth, Data])),
    Id.


-spec create_workflow_schema(aai:auth(), term()) -> binary().
create_workflow_schema(Auth, NameOrData) ->
    {ok, Id} = ?assertMatch({ok, _}, ?CALL(zone, [Auth, NameOrData])),
    Id.


-spec update_workflow_schema(aai:auth(), binary(), map()) -> ok.
update_workflow_schema(Auth, WorkflowId, Data) ->
    ?assertMatch(ok, ?CALL(zone, [Auth, WorkflowId, Data])).


-spec get_workflow_schema(aai:auth(), binary()) -> term().
get_workflow_schema(Auth, AtmWorkflowSchemaId) ->
    {ok, Schema} = ?assertMatch({ok, _}, ?CALL(zone, [Auth, AtmWorkflowSchemaId])),
    Schema.


-spec list_handle_services() -> [binary()].
list_handle_services() ->
    {ok, Services} = ?assertMatch({ok, _}, ?CALL(zone, [aai:root_auth()])),
    Services.


-spec add_user_to_handle_service(binary(), binary(), list()) -> ok.
add_user_to_handle_service(HServiceId, UserId, Privileges) ->
    {ok, UserId} = ?assertMatch({ok, _}, ?CALL(zone, [aai:root_auth(), HServiceId, UserId, Privileges])),
    ok.


-spec remove_user_from_handle_service(binary(), binary()) -> ok.
remove_user_from_handle_service(HServiceId, UserId) ->
    ?assertMatch(ok, ?CALL(zone, [aai:root_auth(), HServiceId, UserId])).


