%%%-------------------------------------------------------------------
%%% @author Bartosz Walkowicz
%%% @copyright (C) 2024 ACK CYFRONET AGH
%%% This software is released under the MIT license
%%% cited in 'LICENSE.txt'.
%%% @end
%%%-------------------------------------------------------------------
%%% @doc
%%% Module responsible for managing code coverage in tests.
%%% @end
%%%-------------------------------------------------------------------
-module(oct_coverage).
-author("Bartosz Walkowicz").

%% API
-export([
    init/1,
    is_enabled/0
]).

-type cover_spec() :: #{
    covered_dirs := [atom()],
    excluded_modules := [module()]
}.


%%====================================================================
%% API
%%====================================================================


%% @doc Initialize coverage configuration if enabled
-spec init(test_config:config()) -> test_config:config().
init(Config) ->
    case is_enabled() of
        true ->
            CoverSpec = load_cover_spec(Config),
            add_service_config(Config, CoverSpec);
        false ->
            Config
    end.


%% @doc Check if coverage is enabled
-spec is_enabled() -> boolean().
is_enabled() ->
    os:getenv("cover") == "true".


%%====================================================================
%% Internal functions
%%====================================================================


%% @private
-spec load_cover_spec(test_config:config()) -> cover_spec().
load_cover_spec(Config) ->
    ProjectRoot = test_config:get_project_root_path(Config),
    {ok, CoverSpec} = file:consult(filename:join(ProjectRoot, "test_distributed/cover_tmp.spec")),

    CoveredDirs = lists:map(fun(DirRelPath) ->
        list_to_atom(filename:join(ProjectRoot, DirRelPath))
    end, kv_utils:get(incl_dirs_r, CoverSpec)),

    #{
        covered_dirs => CoveredDirs,
        excluded_modules => kv_utils:get(excl_mods, CoverSpec)
    }.


%% @private
-spec add_service_config(test_config:config(), cover_spec()) -> test_config:config().
add_service_config(Config, #{covered_dirs := CoveredDirs, excluded_modules := ExcludedModules}) ->
    CoverConfig = [{covered_dirs, CoveredDirs}, {covered_excluded_modules, ExcludedModules}],

    case infer_git_repo_name() of
        "onepanel" ->
            ConfigWithOzPanel = test_config:add_envs(Config, oz_panel, oz_panel, CoverConfig),
            test_config:add_envs(ConfigWithOzPanel, op_panel, op_panel, CoverConfig);
        "op-worker" ->
            test_config:add_envs(Config, op_worker, op_worker, CoverConfig);
        "oz-worker" ->
            test_config:add_envs(Config, oz_worker, oz_worker, CoverConfig);
        "cluster-manager" ->
            test_config:add_envs(Config, cluster_manager, cluster_manager, CoverConfig)
    end.


%% @private
-spec infer_git_repo_name() -> string().
infer_git_repo_name() ->
    string:trim(os:cmd("basename -s .git `git config --get remote.origin.url`")). 
