%%%-------------------------------------------------------------------
%%% @author Konrad Zemek
%%% @copyright (C) 2018 ACK CYFRONET AGH
%%% This software is released under the MIT license
%%% cited in 'LICENSE.txt'.
%%% @end
%%%-------------------------------------------------------------------
%%% @doc
%%% @end
%%%-------------------------------------------------------------------
-module(rtransfer_link_utils).
-author("Konrad Zemek").

%%%===================================================================
%%% Exports
%%%===================================================================

-export([app_path/2, random_element/1, deep_merge/2, hostname_to_binary/1]).

%%%===================================================================
%%% API
%%%===================================================================

-spec app_path(SubDir :: atom(), string() | [string()]) -> string().
app_path(SubDir, [SP | _] = SubPaths) when is_atom(SubDir), is_list(SP) ->
    SubDirList = atom_to_list(SubDir),
    case code:lib_dir(rtransfer_link, SubDir) of
        {error, bad_name} ->
            case filelib:is_dir(filename:join(["..", SubDirList])) of
                true ->
                    filename:join(["..", SubDirList | SubPaths]);
                _ ->
                    filename:join([SubDirList | SubPaths])
            end;

        Dir ->
            filename:join([Dir | SubPaths])
    end;
app_path(SubDir, SubPath) when is_atom(SubDir), is_list(SubPath) ->
    app_path(SubDir, [SubPath]).

-spec random_element(list(Elem)) -> Elem when Elem :: term().
random_element(List) ->
    Len = length(List),
    Index = rand:uniform(Len),
    lists:nth(Index, List).

-spec deep_merge(map(), map()) -> map().
deep_merge(#{} = LMap, #{} = RMap) ->
    Keys = lists:usort(maps:keys(LMap) ++ maps:keys(RMap)),
    lists:foldl(
      fun(Key, Acc) ->
              case {maps:is_key(Key, LMap), maps:is_key(Key, RMap)} of
                  {true, true} ->
                      Val = merge_value(maps:get(Key, LMap), maps:get(Key, RMap)),
                      maps:put(Key, Val, Acc);
                  {true, false} ->
                      maps:put(Key, maps:get(Key, LMap), Acc);
                  {false, true} ->
                      maps:put(Key, maps:get(Key, RMap), Acc)
              end
      end,
      #{},
      Keys).

-spec hostname_to_binary(rtransfer_link:hostname()) -> binary().
hostname_to_binary(IPAddr) when is_tuple(IPAddr) ->
    hostname_to_binary(inet:ntoa(IPAddr));
hostname_to_binary(Hostname) when is_list(Hostname) ->
    unicode:characters_to_binary(Hostname);
hostname_to_binary(Hostname) when is_binary(Hostname) ->
    Hostname.

%%%===================================================================
%%% Helpers
%%%===================================================================

merge_value(#{} = LMap, #{} = RMap) -> deep_merge(LMap, RMap);
merge_value(_L, R) -> R.
