%%%-------------------------------------------------------------------
%%% @author Konrad Zemek
%%% @copyright (C) 2018 ACK CYFRONET AGH
%%% This software is released under the MIT license
%%% cited in 'LICENSE.txt'.
%%% @end
%%%-------------------------------------------------------------------
%%% @doc
%%% @end
%%%-------------------------------------------------------------------
-module(rtransfer_link_table).
-author("Konrad Zemek").

-behaviour(lbm_kv).

%%%===================================================================
%%% Type definitions
%%%===================================================================

-record(?MODULE, {
           ref :: reference(),
           node :: node(),
           request :: rtransfer_link_request:t()
          }).

-type row() :: #{ref := reference(), node := node(),
                 request := rtransfer_link_request:t()}.

%%%===================================================================
%%% Exports
%%%===================================================================

-export([create/0, attributes/0, handle_conflict/3, nodes/0]).
-export([write/1, read/1, delete/1, map_all_for_node/2]).
-export_type([row/0]).

%%%===================================================================
%%% API
%%%===================================================================

-spec create() -> {atomic, ok} | {aborted, term()}.
create() ->
    case lbm_kv:create(?MODULE) of
        ok -> mnesia:add_table_index(?MODULE, node);
        Err -> Err
    end.

-spec attributes() -> [atom()].
attributes() ->
    record_info(fields, rtransfer_link_table).

-spec handle_conflict(Key :: term(), Local :: term(), Remote :: term()) ->
    {value, term()} | delete | term().
handle_conflict(_Key, Local, _Remote) ->
    Local.

-spec nodes() -> [node()].
nodes() ->
    mnesia:table_info(?MODULE, ram_copies).

-spec write(row()) -> ok.
write(Row) ->
    transaction(fun mnesia:write/1, [map_to_record(Row)]).

-spec read(reference()) -> row() | false.
read(Ref) when is_reference(Ref) ->
    case mnesia:dirty_read(?MODULE, Ref) of
        [Rec] -> record_to_map(Rec);
        [] -> false
    end.

-spec delete(reference()) -> ok.
delete(Ref) when is_reference(Ref) ->
    transaction(fun mnesia:delete/1, [{?MODULE, Ref}]).

-spec map_all_for_node(node(), fun(([map()]) -> [map()])) -> any().
map_all_for_node(Node, Fun) when is_atom(Node) ->
    transaction(
      fun(N, F) ->
              mnesia:write_lock_table(?MODULE),
              Records = mnesia:index_read(?MODULE, N, #?MODULE.node),
              lists:foreach(fun mnesia:delete/1, Records),
              MapsToWrite = F([record_to_map(R) || R <- Records]),
              lists:foreach(fun mnesia:write/1, [map_to_record(M) || M <- MapsToWrite])
      end,
      [Node, Fun]).

%%%===================================================================
%%% Helpers
%%%===================================================================

-spec record_to_map(#?MODULE{}) -> row().
record_to_map(#?MODULE{ref = Ref, node = Node, request = Req}) ->
    #{ref => Ref, node => Node, request => Req}.

-spec map_to_record(row()) -> #?MODULE{}.
map_to_record(#{ref := Ref, node := Node, request := Req}) ->
    #?MODULE{ref = Ref, node = Node, request = Req}.

-spec transaction(fun(), [term()]) -> term().
transaction(Fun, Args) ->
    {atomic, Result} = mnesia:transaction(Fun, Args),
    Result.
