%%%-------------------------------------------------------------------
%%% @author Konrad Zemek
%%% @copyright (C) 2017 ACK CYFRONET AGH
%%% This software is released under the MIT license
%%% cited in 'LICENSE.txt'.
%%% @end
%%%-------------------------------------------------------------------
%%% @doc
%%% @end
%%%-------------------------------------------------------------------
-module(rtransfer_link_quota_manager).
-author("Konrad Zemek").

-behaviour(gen_server).

%%%===================================================================
%%% Type definitions
%%%===================================================================

-type state() :: #{
             requests := #{SpaceId :: binary() => #{reference() => {}}},
             disabled_spaces := #{binary() => {}}
            }.

%%%===================================================================
%%% Exports
%%%===================================================================

-export([start_link/0, register_request/1, update_disabled_spaces/1]).
-export([init/1, handle_info/2, handle_cast/2, code_change/3,
         terminate/2, handle_call/3]).

%%%===================================================================
%%% API
%%%===================================================================

-spec start_link() -> {ok, pid()} | {error, any()}.
start_link() ->
    gen_server2:start_link({local, ?MODULE}, ?MODULE, {}, []).

-spec register_request(rtransfer_link_request:t()) ->
                              rtransfer_link:on_complete_fun() | false.
register_request(Request) ->
    gen_server2:call(?MODULE, {register_request, Request}).

-spec update_disabled_spaces([binary()]) -> ok.
update_disabled_spaces(NewDisabledSpaces) ->
    gen_server2:cast(?MODULE, {update_disabled_spaces, NewDisabledSpaces}).

%%%===================================================================
%%% gen_server callbacks
%%%===================================================================

-spec init(term()) -> {ok, state()}.
init(_) ->
    {ok, #{requests => #{}, disabled_spaces => #{}}}.

-spec handle_call(term(), {pid(), term()}, state()) ->
                         {reply, term(), state()}.
handle_call({register_request, #{ref := Ref} = Request}, _,
            #{disabled_spaces := DisabledSpaces,
              requests := Requests} = State) ->

    SpaceId = maps:get(space_id, Request, null_spaceid()),
    case maps:is_key(SpaceId, DisabledSpaces) of
        true -> {reply, false, State};
        false ->
            OnComplete0 = maps:get(on_complete, Request, fun(_, _) -> ok end),
            OnComplete =
                fun(Rf, Status) ->
                        gen_server2:cast(?MODULE, {unregister_request, Rf, SpaceId}),
                        erlang:apply(OnComplete0, [Rf, Status])
                end,
            NewRequests = maps:update_with(SpaceId,
                                           fun(Rs) -> maps:put(Ref, {}, Rs) end,
                                           #{Ref => {}},
                                           Requests),
            {reply, OnComplete, State#{requests := NewRequests}}
    end.

-spec handle_info(term(), state()) -> {noreply, state()}.
handle_info(_Request, State) ->
    lager:warning("Unknown message: ~p", [_Request]),
    {noreply, State}.

-spec handle_cast(term(), state()) -> {noreply, state()}.
handle_cast({update_disabled_spaces, NewDisabledSpacesList0},
            #{disabled_spaces := DisabledSpaces,
              requests := Requests} = State) ->
    NewDisabledSpacesList = [null_spaceid() | NewDisabledSpacesList0],
    NewDisabledSpaces = maps:from_list([{DS, {}} || DS <- NewDisabledSpacesList]),
    NewlyDisabledSpaces = maps:keys(maps:without(maps:keys(DisabledSpaces),
                                                 NewDisabledSpaces)),

    RequestsToCancel = lists:flatmap(
                         fun(SpaceId) ->
                                 maps:keys(maps:get(SpaceId, Requests, #{}))
                         end,
                         NewlyDisabledSpaces),

    lists:foreach(fun rtransfer_link:cancel/1, RequestsToCancel),

    NewRequests = maps:without(NewlyDisabledSpaces, Requests),

    {noreply, State#{disabled_spaces := NewDisabledSpaces,
                     requests := NewRequests}};

handle_cast({unregister_request, Ref, SpaceId},
            #{requests := Requests} =  State) ->
    lager:debug("Unregistering request ~p", [Ref]),
    NewRequests = maps:update_with(SpaceId,
        fun(Rs) -> maps:remove(Ref, Rs) end, #{}, Requests),
    {noreply, State#{requests := NewRequests}}.

-spec code_change(term(), state(), term()) -> {ok, state()}.
code_change(_, State, _Extra) ->
    {ok, State}.

-spec terminate(term(), state()) -> ok.
terminate(_, _) ->
    ok.

%%%===================================================================
%%% Helpers
%%%===================================================================

-spec null_spaceid() -> binary().
null_spaceid() ->
    <<>>.
