#include "service.hpp"

#include <folly/Bits.h>
#include <monitoring/monitoring.h>

#include "serverSideLink.hpp"
#include "storage.hpp"

namespace folly {
// This is defined to fix linker error in C++14, where static constexpr
// are not explicitly inlined...

// NOLINTNEXTLINE
constexpr std::chrono::nanoseconds detail::Sleeper::kMinYieldingSleep;
}  // namespace folly

namespace rtransfer {

folly::Future<Service::MsgPtr> Service::operator()(MsgPtr request)
{
    if (request->payload_case() != proto::LinkMessage::kPing)
        VLOG(2) << "Request from remote: " << request->DebugString();

    ONE_METRIC_COUNTER_INC("requests.remote.incoming.total");
    ONE_METRIC_COUNTER_INC("requests.remote.incoming.concurrent");

    auto prio = request->payload_case() == proto::LinkMessage::kAcks
                    ? SHAPER_OPS_PRIO
                    : 0;

    auto reqId = request->req_id();
    return via(folly::getCPUExecutor().get(), prio)
        .thenValue(
            [this, request = std::move(request)](auto && /*unit*/) mutable {
                return action(std::move(request));
            })
        .thenTry([reqId = std::move(reqId)](folly::Try<MsgPtr> &&maybeResp) {
            if (maybeResp.hasException()) {
                LOG(WARNING) << "Request failed due to exception: "
                             << maybeResp.exception().what();
                return error(maybeResp.exception().get_exception()->what());
            }

            auto &resp = maybeResp.value();

            if (resp) {
                resp->set_req_id(reqId);
                if (resp->payload_case() != proto::LinkMessage::kPong)
                    VLOG(2) << "Replying to req:\n" << resp->DebugString();
            }

            return folly::makeFuture<MsgPtr>(std::move(resp));
        })
        .ensure([] {
            ONE_METRIC_COUNTER_DEC("requests.remote.incoming.concurrent");
        });
}

folly::Future<Service::MsgPtr> Service::action(MsgPtr request)
{
    switch (request->payload_case()) {
        case proto::LinkMessage::kFetch:
            ONE_METRIC_COUNTER_INC("requests.remote.incoming.fetch");
            return fetch(std::move(request));
        case proto::LinkMessage::kCancel:
            ONE_METRIC_COUNTER_INC("requests.remote.incoming.cancel");
            return cancel(std::move(request));
        case proto::LinkMessage::kAcks:
            return ack(std::move(request));
        case proto::LinkMessage::kPing:
            return ping(std::move(request));
        default:
            ONE_METRIC_COUNTER_INC("requests.remote.incoming.bad_request");
            return error("invalid operation");
    }
}

folly::Future<Service::MsgPtr> Service::fetch(MsgPtr request)
{
    auto reqId = request->fetch().req_id();
    auto reqIdOrigin = request->req_id();
    auto requestedSize = request->fetch().size();
    folly::fbstring transferData = request->fetch().transfer_data();
    return authCache_.authorize(providerId_, std::move(transferData))
        .thenTry([reqId, reqIdOrigin, request = std::move(request)](
                     folly::Try<std::shared_ptr<link_control::proto::Request>>
                         &&maybeAuthResponse) mutable {
            if (maybeAuthResponse.hasException()) {
                LOG(WARNING)
                    << "Authorization for fetch failed due to exception: "
                    << maybeAuthResponse.exception().what();

                return folly::makeFuture<std::unique_ptr<proto::LinkMessage>>(
                    maybeAuthResponse.exception());
            }

            const auto &ar = maybeAuthResponse.value()->auth_response();

            VLOG(2) << "ReqId: " << reqId << " using file_id: " << ar.file_id()
                    << ", storage_id: " << ar.storage_id()
                    << " as source for guid: " << ar.file_guid();

            auto *f = request->mutable_fetch();
            f->set_file_id(ar.file_id());
            f->set_src(ar.storage_id());
            f->set_file_guid(ar.file_guid());

            return folly::makeFuture(std::move(request));
        })
        .thenTry([this](folly::Try<MsgPtr> &&maybeRequest) {
            if (maybeRequest.hasException()) {
                LOG(WARNING)
                    << "Authorization for fetch failed due to exception: "
                    << maybeRequest.exception().what();
                return folly::Future<std::size_t>(maybeRequest.exception());
            }

            return shaperMap_.fetch(std::move(maybeRequest.value()));
        })
        .thenTry([reqId, reqIdOrigin, requestedSize](
                     folly::Try<std::size_t> &&maybeSize) {
            if (maybeSize.hasException()) {
                LOG(WARNING) << "Fetch failed due to exception: "
                             << maybeSize.exception().what();

                auto resp = std::make_unique<proto::LinkMessage>();
                resp->set_error(maybeSize.exception().get_exception()->what());
                resp->set_req_id(reqIdOrigin);
                return resp;
            }

            const auto size = maybeSize.value();

            if (size >= requestedSize)
                return MsgPtr{nullptr};  // no need to correct client

            VLOG(2) << "Sending back read result req_id: " << reqId
                    << ", size: " << size << " bytes";

            auto resp = std::make_unique<proto::LinkMessage>();
            resp->set_total_size(size);
            return resp;
        });
}

folly::Future<Service::MsgPtr> Service::cancel(MsgPtr request)
{
    return shaperMap_.cancel(std::move(request))
        .thenValue([](auto && /*unit*/) {
            auto resp = std::make_unique<proto::LinkMessage>();
            resp->set_done(true);
            return folly::makeFuture<MsgPtr>(std::move(resp));
        });
}

folly::Future<Service::MsgPtr> Service::ack(MsgPtr request)
{
    shaperMap_.ack(std::move(request));
    return folly::makeFuture<MsgPtr>(MsgPtr{nullptr});
}

folly::Future<Service::MsgPtr> Service::ping(MsgPtr /*unused*/)
{
    auto resp = std::make_unique<proto::LinkMessage>();
    resp->mutable_pong();
    return folly::makeFuture<MsgPtr>(std::move(resp));
}

folly::Future<Service::MsgPtr> Service::error(folly::StringPiece description)
{
    auto msg = std::make_unique<proto::LinkMessage>();
    msg->set_error(description.data(), description.size());
    return folly::makeFuture(std::move(msg));
}

}  // namespace rtransfer
