#pragma once

#include <folly/Bits.h>
#include <folly/SpinLock.h>
#include <folly/futures/Future.h>
#include <folly/io/IOBuf.h>
#include <folly/io/IOBufQueue.h>
#include <monitoring/monitoring.h>

#include <cstdint>
#include <memory>

#include "common.hpp"
#include "handleCache.hpp"
#include "serverSideLink.hpp"
#include "storage.hpp"

namespace rtransfer {

/**
 * @c Reader schedules reading data from storage and then handles sending it
 * over data pipelines.
 * It definitely violates single responsibility principle, but the "reading"
 * part is trivial enough so that in fact this class is more a "sender".
 *
 * Sends are distributed over available data pipelines (fetched from @c
 * ServerSideLink::dataConnections ) by a semi-complicated aims to schedule less
 * bytes to send through slow, and more bytes to send through fast connections -
 * this is in a response to actual environments we had where connections between
 * two hosts differed wildly in performance.
 *
 * Not only is the data split between connections, it is also flat-combined as
 * we aim to pack TCP packets as much as we can.
 *
 * Finally, for agility reasons we split connections into 4 even pools, each of
 * which is used for different priorities, e.g. for 16 connections, the first 4
 * would be used for prios 0-64, next 4 for 65-128, etc. The corollary is, the
 * number of connections should be 4*n .
 */
class Reader {
public:
    Reader(StoragesMap &storages, HandleCache &handleCache,
        std::shared_ptr<ServerSideLink> serverLink);

    folly::Future<std::size_t> read(std::uint64_t reqId,
        const folly::fbstring &storageId, const folly::fbstring &fileId,
        const folly::fbstring &fileGuid, std::uint64_t offset, std::size_t size,
        std::uint8_t priority);

private:
    struct SendTask {
        folly::Promise<folly::Unit> promise;
        folly::IOBufQueue buf;
        std::uint64_t reqId;
        std::uint64_t offset;
        std::uint8_t priority;
    };

    static std::unique_ptr<folly::IOBuf> constructDataMsg(std::uint64_t reqId,
        std::uint64_t offset, std::uint8_t part, bool isLastPart,
        std::unique_ptr<folly::IOBuf> buf);

    void startSending();

    void doStartSending();

    folly::fbvector<std::pair<ServerSideLink::Conn, folly::IOBuf *>>
    connsForPrio(folly::fbvector<std::pair<ServerSideLink::Conn,
                     std::unique_ptr<folly::IOBuf>>> &allConnsToData,
        std::uint8_t prio);

    void distributeParts(folly::fbvector<std::pair<ServerSideLink::Conn,
                             std::unique_ptr<folly::IOBuf>>> &connsToData,
        SendTask &sendTask);

    std::vector<folly::Future<folly::Unit>> sendParts(
        folly::fbvector<std::pair<ServerSideLink::Conn,
            std::unique_ptr<folly::IOBuf>>> &connsToData);

    folly::fbvector<SendTask> queuedSendTasks_;
    folly::fbvector<SendTask> usedSendTasks_;
    bool sendInProgress_ = false;
    folly::SpinLock spinLock_;

    StoragesMap &storages_;
    HandleCache &handleCache_;
    std::shared_ptr<ServerSideLink> serverLink_;
};

}  // namespace rtransfer
