#pragma once

#include <folly/Format.h>
#include <folly/Optional.h>
#include <folly/Range.h>

#include <string>

namespace rtransfer {
namespace http {

namespace detail {
constexpr static const char* front =
    "GET /rtransfer HTTP/1.1\n"
    "Host: ";
constexpr static const char* rest =
    "\n"
    "Connection: upgrade\n"
    "Upgrade: clproto\n";
constexpr static const char* response =
    "101\n"
    "Connection: upgrade\n"
    "Upgrade: clproto\n";

constexpr std::size_t len(const char* str)
{
    // NOLINTNEXTLINE
    return (*str != 0) ? 1 + len(str + 1) : 0;
}
}  // namespace detail

/**
 * Prepares a HTTP request for the client to send before "upgrading" to
 * rtransfer connection.
 */
inline std::string makeHeader(folly::StringPiece hostname)
{
    return detail::front + hostname.str() + detail::rest;
}

/**
 * Prepares a HTTP response for the server to send before "upgrading" to
 * rtransfer connection.
 */
inline std::string makeResponse() { return detail::response; }

enum class ParseStatus { more_data, bad_header, ok };

/**
 * As a server, parses HTTP request on a potential rtransfer connection to check
 * if it's one that we expect for the upgrade.
 * @returns Status of request parse, and - if successful - how many bytes did
 * the HTTP request span.
 */
inline std::pair<ParseStatus, std::size_t> parseHeader(
    folly::StringPiece header)
{
    if (header.size() < detail::len(detail::front))
        return {ParseStatus::more_data, 0};

    if (!header.startsWith(detail::front))
        return {ParseStatus::bad_header, 0};

    auto hostPiece = header.subpiece(detail::len(detail::front));
    const std::size_t hostEnd = hostPiece.find('\n');
    if (hostEnd == std::string::npos)
        return {ParseStatus::more_data, 0};

    if (hostEnd < 1 || hostEnd > 253)
        return {ParseStatus::bad_header, 0};

    auto restPiece = hostPiece.subpiece(hostEnd);
    if (restPiece.size() < detail::len(detail::rest))
        return {ParseStatus::more_data, 0};

    if (restPiece.removePrefix(detail::rest))
        return {ParseStatus::ok, restPiece.data() - header.data()};

    return {ParseStatus::bad_header, 0};
}

/**
 * As a client, parses HTTP response on a potential rtransfer connection to
 * check if it's one that we expect for the upgrade.
 * @returns Status of response parse, and - if successful - how many bytes did
 * the HTTP response span.
 */
inline std::pair<ParseStatus, std::size_t> parseResponse(
    const folly::StringPiece resp)
{
    if (resp.size() < detail::len(detail::response))
        return {ParseStatus::more_data, 0};

    auto rest = resp;
    if (rest.removePrefix(detail::response))
        return {ParseStatus::ok, rest.data() - resp.data()};

    return {ParseStatus::bad_header, 0};
}

/**
 * @returns length of the HTTP response (it's static).
 */
constexpr std::size_t responseLen() { return detail::len(detail::response); }

}  // namespace http
}  // namespace rtransfer
