#include "controlService.hpp"

#include <jemalloc/jemalloc.h>
#include <monitoring/monitoring.h>
#include <wangle/channel/StaticPipeline.h>
#include <wangle/codec/LineBasedFrameDecoder.h>
#include <wangle/codec/StringCodec.h>
#include <wangle/service/ServerDispatcher.h>

#include <chrono>

#include "jsonMessageHandler.hpp"
#include "secretManager.hpp"
#include "service.hpp"
#include "stdIOHandler.hpp"
#include "stopBatonHandler.hpp"
#include "storage.hpp"
#include "uuid.hpp"

namespace rtransfer {
namespace link_control {

ControlService::~ControlService()
{
    auto c = clientLinks_.cbegin();
    while (c != clientLinks_.cend()) {
        c->second->closeNow();
        ++c;
    }
    clientLinks_.clear();
}

folly::Future<proto::ResponsePtr> ControlService::operator()(
    proto::RequestPtr request)
{
    auto reqId = request->req_id();
    if (request->is_answer()) {
        VLOG(2) << "Control answer: " << request->DebugString();  // NOLINT

        auto questionIt = questions_.find(reqId);
        if (questionIt == questions_.cend())
            return error("bad answer", reqId);

        questionIt->second->setValue(std::move(request));
        questions_.erase(questionIt);
        return folly::makeFuture<proto::ResponsePtr>(proto::ResponsePtr{});
    }

    VLOG(2) << "Control request: " << request->DebugString();  // NOLINT

    // NOLINTNEXTLINE
    return via(clientExecutor_.get())
        .thenValue(
            [this, request = std::move(request)](auto && /*unit*/) mutable {
                VLOG(2) << "Dispatching control request: " << request->req_id();
                return doAction(std::move(request));
            })
        .thenTry([reqId](folly::Try<proto::ResponsePtr> &&maybeResponse) {
            if (maybeResponse.hasException()) {
                LOG(ERROR) << "Control service doAction for " << reqId
                           << " failed due to: "
                           << maybeResponse.exception().what();
                return error(maybeResponse.exception().what(), reqId);
            }

            auto &response = maybeResponse.value();

            response->set_req_id(reqId);

            if (!response->is_update())
                VLOG(2) << "Control reply: " << response->DebugString();

            return folly::makeFuture<proto::ResponsePtr>(std::move(response));
        });
}

folly::Future<proto::RequestPtr> ControlService::ask(proto::ResponsePtr request)
{
    auto reqId = genUUID();
    auto promise = std::make_shared<folly::Promise<proto::RequestPtr>>();
    auto future = promise->getFuture();
    request->set_req_id(reqId.data(), reqId.size());
    request->set_is_question(true);
    questions_.emplace(std::move(reqId), std::move(promise));
    pipeline_->write(std::move(request));
    return future;
}

void ControlService::registerService(std::weak_ptr<rtransfer::Service> srvc)
{
    services_.emplace_back(std::move(srvc));
}

folly::Future<proto::ResponsePtr> ControlService::doAction(
    proto::RequestPtr request)
{
    const auto reqId = request->req_id();
    VLOG(2) << "Dispatching action: " << request->payload_case();

    try {
        ONE_METRIC_COUNTER_INC("requests.control.total");
        switch (request->payload_case()) {
            case proto::Request::kCreateHelper:
                ONE_METRIC_COUNTER_INC("requests.control.create_helper");
                VLOG(2) << "Dispatching CreateHelper action";
                return createHelper(std::move(request));
            case proto::Request::kConnect:
                ONE_METRIC_COUNTER_INC("requests.control.connect");
                return connect(std::move(request));
            case proto::Request::kFetch:
                ONE_METRIC_COUNTER_INC("requests.control.fetch");
                return fetch(std::move(request));
            case proto::Request::kCancel:
                ONE_METRIC_COUNTER_INC("requests.control.cancel");
                return cancel(std::move(request));
            case proto::Request::kAllowConnection:
                ONE_METRIC_COUNTER_INC("requests.control.allow_connection");
                return allowConnection(std::move(request));
            case proto::Request::kGetMemoryStats:
                return memoryStats(std::move(request));
            default:
                ONE_METRIC_COUNTER_INC("requests.control.bad_request");
                LOG(ERROR) << "Invalid request type: "
                           << request->payload_case();
                return error("bad request", reqId);
        }
    }
    catch (const std::exception &e) {
        LOG(ERROR) << "Control service doAction for " << reqId
                   << " failed due to " << e.what();
        return error(e.what(), reqId);
    }
}

folly::Future<proto::ResponsePtr> ControlService::createHelper(
    proto::RequestPtr req)
{
    const auto &storage_id = req->create_helper().storage_id();
    auto it = storages_.find(storage_id);
    if (it == storages_.end()) {
        LOG_DBG(2) << "Creating new '" << req->create_helper().name()
                   << "' helper for storage: " << storage_id;
        auto helper = helperFactory_(req->create_helper());
        storages_.insert(storage_id, std::move(helper));
        return done();
    }

    std::unordered_map<folly::fbstring, folly::fbstring> params;
    for (const auto &param : req->create_helper().params())
        params.emplace(
            folly::fbstring{param.key()}, folly::fbstring{param.value()});
    params["type"] = req->create_helper().name();
    params["name"] = req->create_helper().name();

    LOG_DBG(2) << "Updating helper parameters for storage: " << storage_id;

    it->second->helper()->updateHelper(params);

    // Invalidate storage file handles cached for client side (download/write)
    for (auto it = clientLinks_.begin(); it != clientLinks_.end(); ++it) {
        it->second->handleCache().invalidateStorage(storage_id);
    }

    // Invalidate storage file handles cached for server side (upload/read)
    for (auto &maybe_srvc : services_) {
        auto srvc = maybe_srvc.lock();
        if (srvc) {
            srvc->handleCache().invalidateStorage(storage_id);
        }
    }

    return done();
}

folly::Future<proto::ResponsePtr> ControlService::error(
    folly::StringPiece description, const std::string &reqId)
{
    auto msg = std::make_unique<proto::Response>();
    msg->set_req_id(reqId);
    msg->mutable_error()->set_description(
        description.data(), description.size());
    return folly::makeFuture(std::move(msg));
}

folly::Future<proto::ResponsePtr> ControlService::done(folly::Unit && /*unit*/)
{
    auto msg = std::make_unique<proto::Response>();
    msg->set_done(true);
    return folly::makeFuture(std::move(msg));
}

folly::Future<proto::ResponsePtr> ControlService::connect(proto::RequestPtr req)
{
    const auto &msg = req->connect();
    if (msg.peer_port() > kMaxPortNumber)
        return error("Bad port number: " + std::to_string(msg.peer_port()),
            req->req_id());

    folly::SocketAddress addr{
        msg.peer_host(), static_cast<std::uint16_t>(msg.peer_port()), true};

    VLOG(2) << "Connecting to " << addr.describe();
    auto client = std::make_shared<client::Client>(
        addr, clientExecutor_, storages_, msg.my_secret(), msg.peer_secret());
    return client->connect()
        .thenValue([this, client, addr](auto && /*unit*/) {
            VLOG(2) << "Connected to " << addr.describe();
            clientLinks_.insert_or_assign(addr.describe(), client);
            client->getCloseFuture().thenValue([=](auto && /*unit*/) {
                auto it = clientLinks_.find(addr.describe());
                if (it == clientLinks_.cend())
                    return;

                LOG(INFO) << "Erasing client " << addr.describe();
                it->second->closeNow();
                clientLinks_.erase(it);

                auto msg = folly::make_unique<proto::Response>();
                msg->mutable_disconnected()->set_connection_id(addr.describe());
                pipeline_->write(std::move(msg));
            });
        })
        .thenValue([addr](auto && /*unit*/) {
            auto resp = std::make_unique<proto::Response>();
            resp->set_connection_id(addr.describe());
            return resp;
        });
}

folly::Future<proto::ResponsePtr> ControlService::fetch(proto::RequestPtr req)
{
    VLOG(2) << "Handling fetch control request";

    const auto &msg = req->fetch();
    const auto reqId = req->req_id();
    const auto fetchReqId = msg.req_id();
    auto it = clientLinks_.find(msg.connection_id());  // NOLINT
    if (it == clientLinks_.cend()) {
        LOG(ERROR) << "Cannot handle request " << reqId
                   << " - no client link found with connection id: "
                   << msg.connection_id();
        return error("no such connection: " + msg.connection_id(), reqId);
    }

    auto notifyCb = [this, reqId](std::uint64_t offset, std::size_t size) {
        notifyAggregator_.notify(reqId, offset, size);
    };

    // Send fetch request to remote rtransfer
    auto requestDescription =
        fmt::format("req(reqId={},file={},offset={},size={})", fetchReqId,
            req->fetch().dest_file_id(), req->fetch().offset(),
            req->fetch().size());

    VLOG(2) << "Sending fetch request " << requestDescription
            << " to rtransfer " << it->first.toStdString();

    auto cancelCb =
        [this, reqId, fetchReqId, connectionId = msg.connection_id(),
            srcStorageId = msg.src_storage_id(),
            destStorageId = msg.dest_storage_id()](const std::string &err) {
            if (err == "canceled") {
                // If this is a 'canceled' error do nothing here
                // It's already canceled
                return folly::makeFuture();
            }

            auto it = clientLinks_.find(connectionId);  // NOLINT
            if (it == clientLinks_.cend()) {
                LOG(ERROR) << "Cannot cancel request " << reqId
                           << " - no client link found with connection id: "
                           << connectionId;
                return folly::makeFuture();
            }

            LOG(WARNING) << "Cancelling request " << fetchReqId
                         << " due to error: " << err;

            return it->second->cancel(fetchReqId, srcStorageId, destStorageId);
        };

    return it->second
        ->fetch(msg.src_storage_id(), msg.src_file_id(), msg.dest_storage_id(),
            msg.dest_file_id(), msg.file_guid(), msg.offset(), msg.size(),
            msg.priority(), msg.req_id(), msg.transfer_data(),
            std::move(notifyCb))
        .thenTry([this, reqId = req->req_id(), fetchReqId, requestDescription,
                     cancelCb = std::move(cancelCb)](
                     folly::Try<std::size_t> maybeWrote)
                     -> folly::Future<std::unique_ptr<proto::Response>> {
            if (maybeWrote.hasException()) {
                DCHECK(maybeWrote.exception().get_exception() != nullptr);

                std::string msg{maybeWrote.exception().get_exception()->what()};

                LOG(ERROR) << "Fetch for request " << requestDescription
                           << " failed due to: " << msg;

                notifyAggregator_.flush(reqId);

                return cancelCb(msg).thenTry(
                    [this, msg, reqId](
                        auto && /*unit*/) { return error(msg, reqId); });
            }

            VLOG(2) << "Fetch succeeded for request " << fetchReqId;

            notifyAggregator_.flush(reqId);

            auto resp = std::make_unique<proto::Response>();
            resp->set_wrote(maybeWrote.value());
            return folly::makeFuture<std::unique_ptr<proto::Response>>(
                std::move(resp));
        });
}

void ControlService::notify(
    folly::fbstring reqId, std::uint64_t offset, std::size_t size)
{
    auto resp = std::make_unique<proto::Response>();
    resp->set_req_id(reqId.data());
    resp->set_is_update(true);
    auto *progress = resp->mutable_progress();
    progress->set_offset(offset);
    progress->set_size(size);
    pipeline_->write(std::move(resp));
}

folly::Future<proto::ResponsePtr> ControlService::cancel(proto::RequestPtr req)
{
    const auto &msg = req->cancel();
    auto it = clientLinks_.find(msg.connection_id());  // NOLINT
    if (it == clientLinks_.cend())
        return error(
            "no such connection: " + msg.connection_id(), req->req_id());

    return it->second
        ->cancel(msg.req_id(), msg.src_storage_id(), msg.dest_storage_id())
        .thenValue(&ControlService::done);
}

folly::Future<proto::ResponsePtr> ControlService::allowConnection(
    proto::RequestPtr req)
{
    const auto &msg = req->allow_connection();
    auto expirationTime = std::chrono::steady_clock::now() +
                          std::chrono::milliseconds{msg.expiration()};
    secretManager_.allowConnection(
        msg.my_secret(), msg.peer_secret(), msg.provider_id(), expirationTime);
    return done();
}

folly::Future<proto::ResponsePtr> ControlService::memoryStats(
    proto::RequestPtr /*req*/)
{
    static auto makeResponse = [](void *m, const char *stats) {
        static_cast<proto::Response *>(m)->set_memory_stats(stats);
    };

    auto resp = std::make_unique<proto::Response>();
    malloc_stats_print(makeResponse, resp.get(), "J");
    return folly::makeFuture(std::move(resp));
}

Pipeline::Ptr newPipeline(ControlService &service, folly::Baton<> &stopBaton)
{
    auto stdIOHandler = std::make_shared<StdIOHandler>();
    auto pipeline = wangle::StaticPipeline<folly::IOBufQueue &,
        proto::ResponsePtr, StdIOHandler, StopBatonHandler,
        wangle::LineBasedFrameDecoder, wangle::StringCodec,
        JSONMessageHandler<proto::Request, proto::Response>,
        wangle::MultiplexServerDispatcher<proto::RequestPtr,
            proto::ResponsePtr>>::create(stdIOHandler,
        StopBatonHandler{stopBaton},
        wangle::LineBasedFrameDecoder{UINT_MAX, true,
            wangle::LineBasedFrameDecoder::TerminatorType::NEWLINE},
        wangle::StringCodec{},
        JSONMessageHandler<proto::Request, proto::Response>{},
        wangle::MultiplexServerDispatcher<proto::RequestPtr,
            proto::ResponsePtr>{&service});

    service.setPipeline(pipeline.get());

    return pipeline;
}

}  // namespace link_control
}  // namespace rtransfer
