#pragma once

#include <folly/io/IOBuf.h>
#include <folly/io/IOBufQueue.h>
#include <wangle/channel/Pipeline.h>
#include <wangle/channel/StaticPipeline.h>
#include <wangle/codec/LengthFieldBasedFrameDecoder.h>
#include <wangle/codec/LengthFieldPrepender.h>
#include <wangle/service/ServerDispatcher.h>
#include <wangle/service/Service.h>

#include "proto/rtransfer_messages.pb.h"
#include "protoHandler.hpp"
#include "serverSideLink.hpp"
#include "service.hpp"

namespace rtransfer {
namespace server {

/**
 * This is a special handler for a "default" Wangle pipeline - a hack.
 * Wangle has (had?) no way to morph a pipeline into another pipeline. This is
 * problematic in our case, where we only learn whether a socket is a control
 * connection or a data connection after we receive some data on the socket,
 * while a @c ServerBootstrap only gives us pipelines of a single set type.
 * That's why all pipelines used by the server are `wangle::DefaultPipeline`,
 * and instead for control connections we use this handler that creates and
 * manages *another* pipeline, this time a proper control one, and passes
 * incoming data into this new pipeline and passes data outgoing from the new
 * pipeline into the basic, `DefaultPipeline`.
 */
class ControlHandler : public wangle::HandlerAdapter<folly::IOBufQueue &,
                           std::unique_ptr<folly::IOBuf>> {
public:
    using MsgPtr = std::unique_ptr<proto::LinkMessage>;
    using Pipeline = wangle::Pipeline<folly::IOBufQueue, MsgPtr>;

    /**
     * Create a handler that manages and hooks into a control pipeline.
     * The control pipeline contains a @c ServerSideLink instance that will be
     * used to associate this control pipeline with data pipelines, and uses a
     * newly-created @c Service instance to handle control requests.
     * @param link a @c ServerSideLink instance that will be used for
     * connections belonging to this control connection's "link".
     * @param serviceFactory Used to create a new @c Service for the pipeline.
     * @param providerId Used as a parameter for `serviceFactory`.
     */
    ControlHandler(std::shared_ptr<ServerSideLink> link,
        ServiceFactory &serviceFactory, folly::fbstring providerId)
    {
        service_ = serviceFactory(link, std::move(providerId));
        controlPipeline_ = wangle::StaticPipeline<folly::IOBufQueue, MsgPtr,
            ControlHandler, wangle::LengthFieldBasedFrameDecoder,
            wangle::LengthFieldPrepender, ProtoHandler<proto::LinkMessage>,
            ServerSideLink,
            wangle::MultiplexServerDispatcher<MsgPtr, MsgPtr>>::create(this,
            wangle::LengthFieldBasedFrameDecoder{2, UINT_MAX, 0, 0, 2},
            wangle::LengthFieldPrepender{2}, ProtoHandler<proto::LinkMessage>{},
            link,
            wangle::MultiplexServerDispatcher<MsgPtr, MsgPtr>{service_.get()});

        controlPipeline_->setOwner(this);
        ctrlCtx_ = controlPipeline_->getContext<ControlHandler>();
    }

    void attachPipeline(Context *ctx) override { rawCtx_ = ctx; }

    void read(Context * /*ctx*/, folly::IOBufQueue &buf) override
    {
        ctrlCtx_->fireRead(buf);
    }

    void readEOF(Context * /*ctx*/) override { ctrlCtx_->fireReadEOF(); }

    void readException(Context * /*ctx*/, folly::exception_wrapper e) override
    {
        ctrlCtx_->fireReadException(std::move(e));
    }

    void transportActive(Context * /*ctx*/) override
    {
        ctrlCtx_->getPipeline()->setTransport(rawCtx_->getTransport());
        ctrlCtx_->fireTransportActive();
    }

    void transportInactive(Context * /*ctx*/) override
    {
        ctrlCtx_->fireTransportInactive();
    }

    folly::Future<folly::Unit> write(
        Context * /*ctx*/, std::unique_ptr<folly::IOBuf> buf) override
    {
        return rawCtx_->fireWrite(std::move(buf));
    }

    folly::Future<folly::Unit> writeException(
        Context * /*ctx*/, folly::exception_wrapper e) override
    {
        return rawCtx_->fireWriteException(std::move(e));
    }

    folly::Future<folly::Unit> close(Context * /*ctx*/) override
    {
        return rawCtx_->fireClose();
    }

private:
    Context *rawCtx_{nullptr};
    Context *ctrlCtx_{nullptr};
    Pipeline::Ptr controlPipeline_;
    std::shared_ptr<Service> service_;
};

}  // namespace server
}  // namespace rtransfer
