%%%-------------------------------------------------------------------
%%% @author Piotr Duleba
%%% @copyright (C) 2020 ACK CYFRONET AGH
%%% This software is released under the MIT license
%%% cited in 'LICENSE.txt'.
%%% @end
%%%-------------------------------------------------------------------
%%% @doc
%%% This file provides functions for making rpc calls to OpWorker Nodes for use during tests.
%%% @end
%%%-------------------------------------------------------------------
-module(opw_test_rpc).
-author("Piotr Duleba").

-include_lib("ctool/include/test/test_utils.hrl").

%% API
-export([
    call/2, call/3,
    insecure_call/2, insecure_call/3,

    call/4, call/5,
    insecure_call/4, insecure_call/5,

    get_env/2,
    get_env/3,
    set_env/3,

    create_session/3, create_session/4,
    build_token_credentials/6,

    get_storages/1,
    storage_describe/2,
    is_storage_imported/2,

    get_user_space_by_name/3,

    get_spaces/1,
    get_space_details/2,
    get_space_local_storages/2,
    get_space_capacity_usage/2,
    get_autocleaning_status/2,
    get_support_size/2,
    get_space_providers/2,
    supports_space/2,
    get_space_support_parameters/2,
    revoke_space_support/2,

    get_provider_id/1,
    get_provider_domain/1,
    get_provider_node_ip/1,
    get_provider_name/1,
    get_provider_eff_users/1,

    get_cert_chain_ders/1,
    gs_protocol_supported_versions/1,

    perform_io_test/3
]).

-define(ATTEMPTS, 120).

-define(CALL(NodeSelector, Args), test_rpc:call(op_worker, NodeSelector, test_rpc_api, ?FUNCTION_NAME, Args)).


%%%===================================================================
%%% API functions
%%%===================================================================


-spec call(oct_background:node_selector(), test_rpc:routine()) ->
    term() | no_return().
call(NodeSelector, Routine) ->
    call(NodeSelector, Routine, infinity).


-spec call(oct_background:node_selector(), test_rpc:routine(), timeout()) ->
    term() | no_return().
call(NodeSelector, Routine, Timeout) ->
    test_rpc:call(op_worker, NodeSelector, Routine, Timeout).


-spec insecure_call(oct_background:node_selector(), test_rpc:routine()) ->
    term() | no_return().
insecure_call(NodeSelector, Routine) ->
    insecure_call(NodeSelector, Routine, infinity).


-spec insecure_call(oct_background:node_selector(), test_rpc:routine(), timeout()) ->
    term() | no_return().
insecure_call(NodeSelector, Routine, Timeout) ->
    test_rpc:insecure_call(op_worker, NodeSelector, Routine, Timeout).


-spec call(oct_background:node_selector(), module(), atom(), [term()]) ->
    term() | no_return().
call(NodeSelector, Module, FunctionName, Args) ->
    call(NodeSelector, Module, FunctionName, Args, infinity).


-spec call(oct_background:node_selector(), module(), atom(), [term()], timeout()) ->
    term() | no_return().
call(NodeSelector, Module, FunctionName, Args, Timeout) ->
    test_rpc:call(op_worker, NodeSelector, Module, FunctionName, Args, Timeout).


-spec insecure_call(oct_background:node_selector(), module(), atom(), [term()]) ->
    term() | no_return().
insecure_call(NodeSelector, Module, FunctionName, Args) ->
    insecure_call(NodeSelector, Module, FunctionName, Args, infinity).


-spec insecure_call(oct_background:node_selector(), module(), atom(), [term()], timeout()) ->
    term() | no_return().
insecure_call(NodeSelector, Module, FunctionName, Args, Timeout) ->
    test_rpc:insecure_call(op_worker, NodeSelector, Module, FunctionName, Args, Timeout).


-spec get_env(oct_background:node_selector(), atom()) -> term() | no_return().
get_env(NodeSelector, Key) ->
    ?CALL(NodeSelector, [Key]).


-spec get_env(oct_background:node_selector(), atom(), term()) -> term().
get_env(NodeSelector, Key, Default) ->
    ?CALL(NodeSelector, [Key, Default]).


-spec set_env(oct_background:node_selector(), atom(), term()) -> ok.
set_env(NodeSelector, Key, Value) ->
    ?assertMatch(ok, ?CALL(NodeSelector, [Key, Value])).


-spec create_session(oct_background:node_selector(), oct_background:entity_id(), tokens:serialized()) ->
    binary().
create_session(NodeSelector, UserId, AccessToken) ->
    {ok, Session} = ?assertMatch({ok, _}, ?CALL(NodeSelector, [UserId, AccessToken])),
    Session.


-spec create_session(oct_background:node_selector(), oct_background:entity_id(), atom(), tokens:serialized()) ->
    binary().
create_session(NodeSelector, UserId, SessMode, AccessToken) ->
    {ok, Session} = ?assertMatch({ok, _}, ?CALL(NodeSelector, [UserId, SessMode, AccessToken])),
    Session.


-spec build_token_credentials(
    oct_background:node_selector(),
    binary(),
    undefined | binary(),
    undefined | ip_utils:ip(),
    undefined | cv_interface:interface(),
    data_access_caveats:policy()
) ->
    term().
build_token_credentials(NodeSelector, AccessToken, ConsumerToken, PeerIp, Interface, DataAccessCaveatsPolicy) ->
    ?CALL(NodeSelector, [AccessToken, ConsumerToken, PeerIp, Interface, DataAccessCaveatsPolicy]).


-spec get_storages(oct_background:node_selector()) -> [binary()].
get_storages(NodeSelector) ->
    {ok, StorageIds} = ?assertMatch({ok, _}, ?CALL(NodeSelector, [])),
    StorageIds.


-spec storage_describe(oct_background:node_selector(), binary()) -> map().
storage_describe(NodeSelector, StorageId) ->
    {ok, StorageDetails} = ?assertMatch({ok, _}, ?CALL(NodeSelector, [StorageId])),
    StorageDetails.


-spec is_storage_imported(oct_background:node_selector(), binary()) -> boolean().
is_storage_imported(NodeSelector, StorageId) ->
    IsImported = ?CALL(NodeSelector, [StorageId]),
    ?assert(is_boolean(IsImported)),
    IsImported.

-spec get_user_space_by_name(oct_background:node_selector(), binary(), binary()) ->
    binary().
get_user_space_by_name(NodeSelector, SpaceName, AccessToken) ->
    {true, SpaceId} = ?assertMatch({true, _}, ?CALL(NodeSelector, [SpaceName, AccessToken]), ?ATTEMPTS),
    SpaceId.


-spec get_spaces(oct_background:node_selector()) -> [binary()].
get_spaces(NodeSelector) ->
    % TODO VFS-6780 - currently, supporting providers are calculated asynchronously
    % (effective relation) and for some time the provider may no be allowed to fetch the space details.
    {ok, SpaceIds} = ?assertMatch({ok, _}, ?CALL(NodeSelector, []), ?ATTEMPTS),
    SpaceIds.


-spec get_space_details(oct_background:node_selector(), binary()) -> map().
get_space_details(NodeSelector, SpaceId) ->
    % TODO VFS-6780 - currently, supporting providers are calculated asynchronously
    % (effective relation) and for some time the provider may no be allowed to fetch the space details.
    {ok, SpaceDocument} = ?assertMatch({ok, _}, ?CALL(NodeSelector, [SpaceId]), ?ATTEMPTS),
    SpaceDocument.


-spec get_space_local_storages(oct_background:node_selector(), binary()) -> [binary()].
get_space_local_storages(NodeSelector, SpaceId) ->
    % TODO VFS-6780 - currently, supporting providers are calculated asynchronously
    % (effective relation) and for some time the provider may no be allowed to fetch the space details.
    {ok, StoragesIds} = ?assertMatch({ok, _}, ?CALL(NodeSelector, [SpaceId]), ?ATTEMPTS),
    StoragesIds.


-spec get_space_capacity_usage(oct_background:node_selector(), binary()) -> integer().
get_space_capacity_usage(NodeSelector, SpaceId) ->
    ?CALL(NodeSelector, [SpaceId]).


-spec get_autocleaning_status(oct_background:node_selector(), binary()) -> map().
get_autocleaning_status(NodeSelector, SpaceId) ->
    AutocleaningStatus = ?CALL(NodeSelector, [SpaceId]),
    ?assert(is_map(AutocleaningStatus)),
    AutocleaningStatus.


-spec get_support_size(oct_background:node_selector(), binary()) -> integer().
get_support_size(NodeSelector, SpaceId) ->
    % TODO VFS-6780 - currently, supporting providers are calculated asynchronously
    % (effective relation) and for some time the provider may no be allowed to fetch the space details.
    {ok, SupportSize} = ?assertMatch({ok, _}, ?CALL(NodeSelector, [SpaceId]), ?ATTEMPTS),
    SupportSize.


-spec get_space_providers(oct_background:node_selector(), binary()) -> [binary()].
get_space_providers(NodeSelector, SpaceId) ->
    % TODO VFS-6780 - currently, supporting providers are calculated asynchronously
    % (effective relation) and for some time the provider may no be allowed to fetch the space details.
    {ok, ProviderIds} = ?assertMatch({ok, _}, ?CALL(NodeSelector, [SpaceId]), ?ATTEMPTS),
    ProviderIds.


-spec supports_space(oct_background:node_selector(), binary()) -> boolean().
supports_space(NodeSelector, SpaceId) ->
    ?CALL(NodeSelector, [SpaceId]).


-spec get_space_support_parameters(oct_background:node_selector(), binary()) ->
    support_parameters:record().
get_space_support_parameters(NodeSelector, SpaceId) ->
    {ok, SupportParameters} = ?assertMatch({ok, _}, ?CALL(NodeSelector, [SpaceId])),
    SupportParameters.


-spec revoke_space_support(oct_background:node_selector(), binary()) -> ok.
revoke_space_support(NodeSelector, SpaceId) ->
    ?assertMatch(ok, ?CALL(NodeSelector, [SpaceId])).


-spec get_provider_id(oct_background:node_selector()) -> binary().
get_provider_id(NodeSelector) ->
    ?CALL(NodeSelector, []).


-spec get_provider_domain(oct_background:node_selector()) -> binary().
get_provider_domain(NodeSelector) ->
    ?CALL(NodeSelector, []).


-spec get_provider_node_ip(oct_background:node_selector()) -> binary().
get_provider_node_ip(NodeSelector) ->
    ?CALL(NodeSelector, []).


-spec get_provider_name(oct_background:node_selector()) -> binary().
get_provider_name(NodeSelector) ->
    {ok, ProviderName} = ?assertMatch({ok, _}, ?CALL(NodeSelector, [])),
    ProviderName.


-spec get_provider_eff_users(oct_background:node_selector()) -> [binary()].
get_provider_eff_users(NodeSelector) ->
    {ok, Users} = ?assertMatch({ok, _}, ?CALL(NodeSelector, [])),
    Users.


-spec get_cert_chain_ders(oct_background:node_selector()) -> [public_key:der_encoded()].
get_cert_chain_ders(NodeSelector) ->
    ?CALL(NodeSelector, []).


-spec gs_protocol_supported_versions(oct_background:node_selector()) -> [integer()].
gs_protocol_supported_versions(NodeSelector) ->
    ?CALL(NodeSelector, []).


-spec perform_io_test(oct_background:node_selector(), binary(), binary()) -> ok | error.
perform_io_test(NodeSelector, Path, AccessToken) ->
    ?CALL(NodeSelector, [Path, AccessToken]).
